require "prototypes.globals"

-- debug logging
function debug(message)
	if DSB.LOGGING then log(message) end
end

function multiply_colour(colour, factor)
	return {colour.r*factor, colour.g*factor, colour.b*factor, colour.a}
end

-- make a new stacked item
function create_stacked_item(item_name, graphic_path)
	if not data.raw.item[item_name] then
		log("ERROR: DSB asked to create stacks from a non-existent item ("..item_name..")")
		return
	end
	local temp_icons, stacked_icons, this_fuel_category, this_fuel_acceleration_multiplier, this_fuel_top_speed_multiplier, this_fuel_value, this_fuel_emissions_multiplier
	if graphic_path then
		stacked_icons = { { icon = graphic_path } }
	else
		if data.raw.item[item_name].icon then
			temp_icons = { { icon = data.raw.item[item_name].icon } }
			log("WARNING: DSB creating layered stack icon ("..item_name.."), this is 4x more rendering effort than a custom icon")
		elseif data.raw.item[item_name].icons then
			temp_icons = data.raw.item[item_name].icons
			log("WARNING: DSB creating layers-of-layers stack icon ("..item_name.."), this is "..1+(#temp_icons*3).."x more rendering effort than a custom icon")
		else
			log("ERROR: DSB asked to create stacks for item with no icon properties ("..item_name..")")
			return
		end
		stacked_icons = { { icon = "__DeadlockStacking__/graphics/blank.png" } }
		for i = 1, -1, -1 do
			for _,layer in pairs(temp_icons) do
				layer.shift = {0, i*3}
				layer.scale = 0.85
				table.insert(stacked_icons, table.deepcopy(layer))
			end
		end
	end
	if data.raw.item[item_name].fuel_value then
		this_fuel_category = data.raw.item[item_name].fuel_category
		this_fuel_acceleration_multiplier = data.raw.item[item_name].fuel_acceleration_multiplier
		this_fuel_top_speed_multiplier = data.raw.item[item_name].fuel_top_speed_multiplier
		this_fuel_emissions_multiplier = data.raw.item[item_name].fuel_emissions_multiplier
		-- great, the fuel value is a string, with SI units. how very easy to work with
		this_fuel_value = (tonumber(string.match(data.raw.item[item_name].fuel_value, "%d+")) * DSB.STACK_SIZE) .. string.match(data.raw.item[item_name].fuel_value, "%a+")
	end
	local menu_order = string.format("%03d",DSB.GLOBAL_ORDER)
	DSB.GLOBAL_ORDER = DSB.GLOBAL_ORDER + 1
	data:extend({
		{
			type = "item",
			name = DSB.ITEM_PREFIX..item_name,
			localised_name = {"item-name.deadlock-stacking-stack", {"item-name."..item_name}, DSB.STACK_SIZE},
			icons = stacked_icons,
			icon_size = 32,
			stack_size = math.floor(data.raw.item[item_name].stack_size/DSB.STACK_SIZE),
			flags = { "goes-to-main-inventory" },
			group = "intermediate-products",
			subgroup = "stacks",
			order = menu_order,
			allow_decomposition = false,
			fuel_category = this_fuel_category,
			fuel_acceleration_multiplier = this_fuel_acceleration_multiplier,
			fuel_top_speed_multiplier = this_fuel_top_speed_multiplier,
			fuel_emissions_multiplier = this_fuel_emissions_multiplier,
			fuel_value = this_fuel_value,
		}
	})
	debug("DSB: Created stacked item: "..item_name)
end

-- make stacking/unstacking recipes for a base item
function create_stacking_recipes(item_name)
	if not data.raw.item[item_name] then
		log("ERROR: DSB asked to create recipes from a non-existent item ("..item_name..")")
		return
	end
	if not data.raw.item[DSB.ITEM_PREFIX..item_name] then
		log("ERROR: DSB asked to create recipes from a non-existent stacked item ("..DSB.ITEM_PREFIX..item_name..")")
		return
	end
	-- icon prep
	local menu_order = string.format("%03d",DSB.GLOBAL_ORDER)
	DSB.GLOBAL_ORDER = DSB.GLOBAL_ORDER + 1
	local base_icon = data.raw.item[DSB.ITEM_PREFIX..item_name].icon
	local base_icons = data.raw.item[DSB.ITEM_PREFIX..item_name].icons
	if not base_icons then
		base_icons = { { icon = base_icon } }
	end
	local temp_icons = table.deepcopy(base_icons)
	table.insert(temp_icons, 
		{
			icon = "__DeadlockStacking__/graphics/arrow-d.png",
			scale = 0.5,
		}
	)
	-- stacking
	data:extend({
		{
		type = "recipe",
		name = DSB.RECIPE_PREFIX.."stack-"..item_name,
		localised_name = {"recipe-name.deadlock-stacking-stack", {"item-name."..item_name}},
		category = "stacking",
		group = "intermediate-products",
		subgroup = "stacks",
		order = menu_order.."[a]",
		enabled = false,
		allow_decomposition = false,
		ingredients = { {item_name, DSB.STACK_SIZE} },
		result = DSB.ITEM_PREFIX..item_name,
		result_count = 1,
		energy_required = DSB.CRAFT_TIME,
		icon_size = 32,
		icons = temp_icons,
		hidden = true,
		allow_as_intermediate = false,
		},
	})
	-- unstacking
	temp_icons = table.deepcopy(base_icons)
	table.insert(temp_icons, 
		{
			icon = "__DeadlockStacking__/graphics/arrow-u.png",
			scale = 0.5,
		}
	)
	data:extend({
		{
		type = "recipe",
		name = DSB.RECIPE_PREFIX.."unstack-"..item_name,
		localised_name = {"recipe-name.deadlock-stacking-unstack", {"item-name."..item_name}},
		category = "unstacking",
		group = "intermediate-products",
		subgroup = "stacks",
		order = menu_order.."[b]",
		enabled = false,
		allow_decomposition = false,
		ingredients = { {DSB.ITEM_PREFIX..item_name, 1} },
		result = item_name,
		result_count = DSB.STACK_SIZE,
		energy_required = DSB.CRAFT_TIME,
		icon_size = 32,
		icons = temp_icons,
		hidden = false,
		allow_as_intermediate = false,
		},
	})
	debug("DSB: Created recipes: "..item_name)
end


-- make the stacking recipes depend on a technology
function add_stacks_to_tech(item_name, target_technology)
	if not target_technology then
		debug("DSB: Skipping technology insert, no tech specified ("..item_name..")")
		return
	end
	if not data.raw.recipe[DSB.RECIPE_PREFIX.."stack-"..item_name] then
		log("ERROR: DSB asked to use non-existent stacking recipe ("..item_name..")")
		return
	end
	if not data.raw.recipe[DSB.RECIPE_PREFIX.."unstack-"..item_name] then
		log("ERROR: DSB asked to use non-existent unstacking recipe ("..item_name..")")
		return
	end
	if not data.raw.technology[target_technology] then
		log("ERROR: DSB asked to use non-existent technology ("..target_technology..")")
		return
	end
	-- request by orzelek - remove previous recipe unlocks if we're re-adding something that was changed by another mod
	for i,effect in ipairs(data.raw.technology[target_technology].effects) do
		if effect.type == "unlock-recipe" and effect.recipe == DSB.RECIPE_PREFIX.."stack-"..item_name then
			table.remove(data.raw.technology[target_technology].effects, i)
			debug("DSB: Removed previous stacking recipe unlock ("..item_name..")")
			break
		end
	end
	for i,effect in ipairs(data.raw.technology[target_technology].effects) do
		if effect.type == "unlock-recipe" and effect.recipe == DSB.RECIPE_PREFIX.."unstack-"..item_name then
			table.remove(data.raw.technology[target_technology].effects, i)
			debug("DSB: Removed previous unstacking recipe unlock ("..item_name..")")
			break
		end
	end
	-- insert stacking recipe
	table.insert(data.raw.technology[target_technology].effects,
		{
		type = "unlock-recipe",
		recipe = DSB.RECIPE_PREFIX.."stack-"..item_name,
		}
	)
	-- insert unstacking recipe
	table.insert(data.raw.technology[target_technology].effects,
		{
		type = "unlock-recipe",
		recipe = DSB.RECIPE_PREFIX.."unstack-"..item_name,
		}
	)
	debug("DSB: Modified technology: "..target_technology)
end

-- public stuff
deadlock_stacking = {}
function deadlock_stacking.create(item_name, graphic_path, target_tech)
	debug("DSB: importing mod item ("..item_name..") ...")
	create_stacked_item(item_name, graphic_path)
	create_stacking_recipes(item_name)
	add_stacks_to_tech(item_name, target_tech)
end
function deadlock_stacking.reset()
    for i=1,DSB.TIERS do
        local e = data.raw.technology[DSB.TECH_PREFIX..i].effects
        if e then 
            while e[2] do table.remove(e,2) end
        end
    end
    log("DSB: Technologies cleared.")
end
function deadlock_stacking.remove(target_tech)
    for i=1,DSB.TIERS do
        local e = data.raw.technology[DSB.TECH_PREFIX..i].effects
        if e then 
            local j = 2
            while e[j] do
                if e[j].type == "unlock-recipe" and string.find(e[j].recipe, target_tech, 1, true) then
                    log("DSB: Recipe "..e[j].recipe.." cleared.")
                    table.remove(e,j)
                else
                    j = j + 1
                end
            end
        end
    end
end
