require "prototypes.globals"

-- debug logging
function debug(message)
	if LOGGING then log(message) end
end

function multiply_colour(colour, factor)
	return {colour.r*factor, colour.g*factor, colour.b*factor, colour.a}
end

-- make a new stacked item
function create_stacked_item(item_name, graphic_path)
	if not data.raw.item[item_name] then
		log("ERROR: DSB asked to create stacks from a non-existent item ("..item_name..")")
		return
	end
	local these_icons, this_fuel_category, this_fuel_acceleration_multiplier, this_fuel_top_speed_multiplier, this_fuel_value
	if graphic_path then
		-- no way to check file existence, so, crash, whatever
		these_icons = {	{ icon = graphic_path }	}
	else
		if not data.raw.item[item_name].icon then
			log("ERROR: DSB asked to create stacks for item with no icon and no custom sprite ("..item_name..")")
			return
		end
		these_icons = {
			{
				icon = "__DeadlockStacking__/graphics/blank.png",
				shift = {0, 0},
				scale = 1.0,
			},
			{
				icon = data.raw.item[item_name].icon,
				shift = {0, 3},
				scale = 0.85,
			},
			{
				icon = data.raw.item[item_name].icon,
				shift = {0, 0},
				scale = 0.85,
			},
			{
				icon = data.raw.item[item_name].icon,
				shift = {0, -3},
				scale = 0.85,
			},
		}
	end
	if data.raw.item[item_name].fuel_value then
		this_fuel_category = data.raw.item[item_name].fuel_category
		this_fuel_acceleration_multiplier = data.raw.item[item_name].fuel_acceleration_multiplier
		this_fuel_top_speed_multiplier = data.raw.item[item_name].fuel_top_speed_multiplier
		-- oh great, the fuel value is a string, with SI units. how very easy to work with
		this_fuel_value = (tonumber(string.match(data.raw.item[item_name].fuel_value, "%d+")) * STACK_SIZE) .. string.match(data.raw.item[item_name].fuel_value, "%a+")
	end
	local menu_order = string.format("%03d",GLOBAL_ORDER)
	GLOBAL_ORDER = GLOBAL_ORDER + 1
	data:extend({
		{
			type = "item",
			name = ITEM_PREFIX..item_name,
			localised_name = {"item-name.deadlock-stacking-stack", {"item-name."..item_name}, STACK_SIZE},
			icons = these_icons,
			icon_size = 32,
			stack_size = math.floor(data.raw.item[item_name].stack_size/STACK_SIZE),
			flags = { "goes-to-main-inventory" },
			group = "intermediate-products",
			subgroup = "stacks",
			order = menu_order,
			allow_decomposition = false,
			fuel_category = this_fuel_category,
			fuel_acceleration_multiplier = this_fuel_acceleration_multiplier,
			fuel_top_speed_multiplier = this_fuel_top_speed_multiplier,
			fuel_value = this_fuel_value,
		}
	})
	debug("DSB: Created stacked item: "..item_name)
end

-- make stacking/unstacking recipes for a base item
function create_stacking_recipes(item_name)
	if not data.raw.item[item_name] then
		log("ERROR: DSB asked to create recipes from a non-existent item ("..item_name..")")
		return
	end
	if not data.raw.item[ITEM_PREFIX..item_name] then
		log("ERROR: DSB asked to create recipes from a non-existent stacked item ("..ITEM_PREFIX..item_name..")")
		return
	end
	-- icon prep
	local menu_order = string.format("%03d",GLOBAL_ORDER)
	GLOBAL_ORDER = GLOBAL_ORDER + 1
	local base_icon = data.raw.item[ITEM_PREFIX..item_name].icon
	local base_icons = data.raw.item[ITEM_PREFIX..item_name].icons
	if not base_icons then
		base_icons = { { icon = base_icon } }
	end
	local temp_icons = table.deepcopy(base_icons)
	table.insert(temp_icons, 
		{
			icon = "__DeadlockStacking__/graphics/arrow-d.png",
			scale = 0.667,
		}
	)
	-- stacking
	data:extend({
		{
		type = "recipe",
		name = RECIPE_PREFIX.."stack-"..item_name,
		localised_name = {"recipe-name.deadlock-stacking-stack", {"item-name."..item_name}},
		category = "stacking",
		group = "intermediate-products",
		subgroup = "stacks",
		order = menu_order.."[a]",
		enabled = false,
		allow_decomposition = false,
		ingredients = { {item_name, STACK_SIZE} },
		result = ITEM_PREFIX..item_name,
		result_count = 1,
		energy_required = CRAFT_TIME,
		icon_size = 32,
		icons = temp_icons,
		hidden = true,
		allow_as_intermediate = false,
		},
	})
	-- unstacking
	temp_icons = table.deepcopy(base_icons)
	table.insert(temp_icons, 
		{
			icon = "__DeadlockStacking__/graphics/arrow-u.png",
			scale = 0.667,
		}
	)
	data:extend({
		{
		type = "recipe",
		name = RECIPE_PREFIX.."unstack-"..item_name,
		localised_name = {"recipe-name.deadlock-stacking-unstack", {"item-name."..item_name}},
		category = "unstacking",
		group = "intermediate-products",
		subgroup = "stacks",
		order = menu_order.."[b]",
		enabled = false,
		allow_decomposition = false,
		ingredients = { {ITEM_PREFIX..item_name, 1} },
		result = item_name,
		result_count = STACK_SIZE,
		energy_required = CRAFT_TIME,
		icon_size = 32,
		icons = temp_icons,
		hidden = false,
		allow_as_intermediate = false,
		},
	})
	debug("DSB: Created recipes: "..item_name)
end


-- make the stacking recipes depend on a technology
function add_stacks_to_tech(item_name, target_technology)
	if not target_technology then
		debug("DSB: Skipping technology insert, no tech specified ("..item_name..")")
		return
	end
	if not data.raw.recipe[RECIPE_PREFIX.."stack-"..item_name] then
		log("ERROR: DSB asked to use non-existent stacking recipe for tech ("..target_technology..")")
		return
	end
	if not data.raw.recipe[RECIPE_PREFIX.."unstack-"..item_name] then
		log("ERROR: DSB asked to use non-existent unstacking recipe for tech ("..target_technology..")")
		return
	end
	if not data.raw.technology[target_technology] then
		log("ERROR: DSB asked to use non-existent technology ("..target_technology..")")
		return
	end
	table.insert(data.raw.technology[target_technology].effects,
		{
		type = "unlock-recipe",
		recipe = RECIPE_PREFIX.."stack-"..item_name,
		}
	)
	table.insert(data.raw.technology[target_technology].effects,
		{
		type = "unlock-recipe",
		recipe = RECIPE_PREFIX.."unstack-"..item_name,
		}
	)
	debug("DSB: Modified technology: "..target_technology)
end

-- public stuff
deadlock_stacking = {}
function deadlock_stacking.create(item_name, graphic_path, target_tech)
	debug("DSB: importing mod item ("..item_name..") ...")
	create_stacked_item(item_name, graphic_path)
	create_stacking_recipes(item_name)
	add_stacks_to_tech(item_name, target_tech)
end